# Testing

In order to test if a content pasted from any external application is transformed correctly by the editor the test itself needs to use data which is put into the native clipboard by that application. For test purposes, such data is stored in a single file called a fixture file.

The fixture file is usually an HTML file containing HTML content which was fetched from the native browser `dataTransfer` object (`dataTransfer.getData( 'html/text' )`). This ensures that the fixture file provides exactly same data as a real use scenario.

## Fixture files

_For example files see `_data/basic-style/bold-within-text/`_.

Fixture files are grouped per feature (which usually corresponds to editor plugins, for example `basic-styles`, `list`, etc). All fixtures are stored in `_data/feature-name/` directory (for example `_data/basic-style/`). Each feature (which will be called a *group*) has a separate folder per fixture. Each fixture is used to create one normalization and one integration test.

Each fixture folder contains:

- original input file - `bold-with-text.docx`
- input fixture - `input.word2016.html`
- normalized output fixture - `normalized.word2016.html`
- model output fixture - `model.word2016.html`

In some cases, different browsers produces different input data. For such situations, additional fixtures are stored. For example if input data is different for Safari, additional `input.safari.word2016.html` file will be present in the fixture directory.

## Tests group index

_For example file see `_data/basic-style/index.js`_.

Each group of fixtures contains an index file (`index.js` in a group folder – e.g. `_data/basic-styles/index.js`). Its purpose is to import all fixture files from the group and expose them for further use. An index file has the following structure:

```js
// Import default/generic fixtures.
// Input fixtures.
import boldWithinText from './bold-within-text/input.word2016.html';

// Expected normalized fixtures.
import boldWithinTextNormalized from './bold-within-text/normalized.word2016.html';

// Expected model fixtures.
import boldWithinTextModel from './bold-within-text/model.word2016.html';

// Export imported generic fixtures for future use.
export const fixtures = {
	input: {
		boldWithinText
	},
	normalized: {
		boldWithinText: boldWithinTextNormalized
	},
	model: {
		boldWithinText: boldWithinTextModel
	}
};
```

Such a structure exports generic fixtures (the ones which are the same for more than one browser and will be used if no browser specific fixtures are present).

Index files must also export browser specific fixtures. In the simplest case if there are none, it exports an empty object:

```js
export browserFixtures = {};
```

If there are any browser specific fixtures, they are exported in a similar manner to generic ones (apart from being grouped by a browser):

```js
// Export imported browser-specific fixtures for future use.
export const browserFixtures = {
	safari: {
		input: {
			boldWithinText: boldWithinTextSafari
		},
		normalized: {
			boldWithinText: boldWithinTextNormalizedSafari
		},
		model: {
			boldWithinText: boldWithinTextModelSafari
		}
	}
};
```

### What if only input or one of the expected output fixtures are different for specific browser? Could fixtures be mixed?

There are cases when only some fixtures differ for a given browser. In such cases browser fixtures export reuses generic fixtures:

```js
// Export imported browser-specific fixtures for future use.
export const browserFixtures = {
	safari: {
		input: {
			boldWithinText: boldWithinText // generic
		},
		normalized: {
			boldWithinText: boldWithinTextNormalizedSafari // Safari specific
		},
		model: {
			boldWithinText: boldWithinTextModel // generic
		}
	}
};
```

## Fixtures aggregation

_See `_utils/fixtures.js`_.

All group index files are aggregated in the `fixtures` util (`_utils/fixtures.js`) and exposed for tests in a single `fixtures` and `browserFixtures` objects:

```js
// Import fixtures.
import { fixtures as basicStyles, browserFixtures as basicStylesBrowser } from '../_data/basic-styles/index.js';

// Generic fixtures.
export const fixtures = {
	'basic-styles': basicStyles
};

// Browser specific fixtures.
export const browserFixtures = {
	'basic-styles': basicStylesBrowser
};
```

## Tests generation

_See `data/normalization.js` and `data/integration.js`_.

Tests based on fixture files are generated by the special util function `generateTests()` (see `_utils/utils.js`). This function is specifically designed to generate `normalization` (see `data/normalization.js`) or `integration` (see `data/integration.js`) tests using provided fixtures group, for example:

```js
generateTests( {
	input: 'basic-styles', // Group name.
	type: 'integration', // Tests type (integration or normalization).
	browsers: [ 'chrome', 'firefox', 'safari', 'edge' ], // For which browsers generate tests.
	editorConfig: { // Editor config which will be used during editor creation which is used in tests.
		plugins: [ Clipboard, Paragraph, Heading, Bold, Italic, Underline, Strikethrough, PasteFromOffice ]
	},
	skip: { // Names of fixtures which tests should be skipped (object `key` is the name of the browser for which to skip tests).
		safari: [ 'italicStartingText', 'multipleStylesSingleLine', 'multipleStylesMultiline' ] // Skip due to spacing issue (#13).
	}
} );
```

## Adding new tests

### To an existing group

1. Create new fixtures directory in a group to which you plan to add tests (e.g. `_data/link/new-use-case/`).
2. Add all necessary fixture files to the above directory:
	* original input file - `new-use-case.docx`
	* input fixture - `input.word2016.html` (to acquire clipboard data you may use `integration.html` manual test which prints `text/html` on paste)
	* normalized output fixture - `normalized.word2016.html`
	* model output fixture - `model.word2016.html`
	* any browser specific fixtures
3. Add new fixtures to group `index.js` file.

That's all, added fixtures will be now used to generate normalization and integration test.

### To a new group

1. Create new group directory, for example `_data/new-group/`.
2. Create new fixtures directories (one per input fixture file) in `_data/new-group/`, each containing:
	* original input file - `new-use-case.docx`
	* input fixture - `input.word2016.html` (to acquire clipboard data you may use `integration.html` manual test which prints `text/html` on paste)
	* normalized output fixture - `normalized.word2016.html`
	* model output fixture - `model.word2016.html`
	* any browser specific fixtures
3. Create group `index.js` file (`_data/new-group/index.js`) importing all necessary fixtures.
4. Add new group to fixtures util `_utils/fixtures.js`:

	```js
	// Import fixtures.
	import { fixtures as newGroup, browserFixtures as newGroupBrowser } from '../_data/new-group/index.js';

	// Generic fixtures.
	export const fixtures = {
		'new-group': newGroup
	};

	// Browser specific fixtures.
	export const browserFixtures = {
		'new-group': newGroupBrowser
	};
	```

5. Add `generateTests()` function call in `data/normalization.js` to generate normalization and in `data/integration.js` to generate integration tests:

	```js
	// normalization.js
	generateTests( {
		input: 'new-group',
		type: 'normalization',
		browsers: [ 'chrome', 'firefox', 'safari', 'edge' ]
		editorConfig: { ... }
	} );

	// integration.js
	generateTests( {
		input: 'new-group',
		type: 'integration',
		browsers: [ 'chrome', 'firefox', 'safari', 'edge' ]
		editorConfig: { ... }
	} );
	```
